<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Blog;
use Illuminate\Support\Facades\Log;
class BlogController extends Controller
{
    // ✅ Show all blogs
    public function index()
    {
        $blogs = Blog::all();
        return response()->json($blogs);
    }

    // ✅ Show create page (Blade form)
    public function create()
    {
        return view('admin.blog.BlogCreate');
    }

  public function store(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:100',
        'title' => 'required|string|max:255',
        'descriptions' => 'required|array|min:1',
        'descriptions.*' => 'required|string',
        'image' => 'required|image|mimes:jpg,jpeg,png|max:2048',
        'status' => 'required|boolean',
    ]);

    try {
        // ✅ Image Upload
        $imageUrl = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $imageUrl = url('storage/' . $imagePath);
        }

        // ✅ Create Blog
        $blog = Blog::create([
            'name' => $request->name,
            'title' => $request->title,
            'descriptions' => json_encode($request->descriptions), // store as JSON
            'image' => $imageUrl,
            'status' => $request->status,
        ]);

        return response()->json([
            'message' => 'Blog created successfully',
            'data' => $blog
        ], 201);

    } catch (\Exception $e) {
        Log::error('Blog creation failed: ' . $e->getMessage());
        return response()->json([
            'message' => 'Server error',
            'error' => $e->getMessage()
        ], 500);
    }
}



    // ✅ Show single blog
    public function show($id)
    {
        $blog = Blog::findOrFail($id);
        return response()->json($blog);
    }

    // ✅ Edit blog (Blade page)
    public function edit($id)
    {
        $blog = Blog::findOrFail($id);
       return response()->json($blog);
    }

   public function update(Request $request, $id)
{
    $request->validate([
        'name' => 'required|string|max:100',
        'title' => 'required|string|max:255',
        'descriptions' => 'required|array|min:1',
        'descriptions.*' => 'required|string',
        'image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048', // ✅ fixed
        'status' => 'required|boolean',
    ]);

    try {
        $blog = Blog::findOrFail($id);

        $imageUrl = $blog->image;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $imageUrl = url('storage/' . $imagePath);
        }

        $blog->update([
            'name' => $request->name,
            'title' => $request->title,
            'descriptions' => $request->descriptions, // ✅ no need to json_encode if model casts it
            'image' => $imageUrl,
            'status' => $request->status,
        ]);

        return response()->json([
            'message' => 'Blog updated successfully',
            'data' => $blog
        ]);
    } catch (\Exception $e) {
        Log::error('Blog update failed: ' . $e->getMessage());
        return response()->json([
            'message' => 'Server error',
            'error' => $e->getMessage()
        ], 500);
    }
}


    // ✅ Delete blog
    public function destroy(Request $request)
    {
        $id = $request->id;
        $brand = Blog::findOrFail($id);
        $brand->delete();

        return response()->json([
            'message' => 'Brand deleted successfully'
        ]);
    }

    // ✅ Toggle status
    public function updateStatus(Request $request)
    {
        $request->validate([
            'id'     => 'required|integer|exists:blogs,id',
            'status' => 'required|boolean',
        ]);

        $blog = Blog::findOrFail($request->id);
        $blog->status = $request->status;
        $blog->save();

        return response()->json(['message' => 'Status updated successfully']);
    }
}