<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Brand;

class BrandController extends Controller
{
    // ✅ Show create page
    public function brandCreatePage(Request $request)
    {
        return view('admin.brand.BrandCreate');
    }

    // ✅ Get all brands
    public function brandList(Request $request)
    {
        $brands = Brand::with('category')->get(); // category relation সহ
        return response()->json($brands);
    }

    // ✅ Store new brand
    public function storeBrand(Request $request)
    {
        $request->validate([
            'name'        => 'required|string|max:100',
            'image'       => 'required|image|mimes:jpg,jpeg,png|max:2048',
            'status'      => 'required|boolean',
            'category_id' => 'required|integer|exists:categories,id',
        ]);

        $imageUrl = null;

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('image', 'public');
            $imageUrl = url('storage/' . $imagePath); // ✅ full URL
        }

        $brand = Brand::create([
            'name'        => $request->input('name'),
            'image'       => $imageUrl,
            'status'      => $request->input('status'),
            'category_id' => $request->input('category_id'),
        ]);

        return response()->json([
            'message' => 'Brand created successfully',
            'data'    => $brand
        ], 201);
    }

    // ✅ Edit single brand
    public function editBrand(Request $request, $id)
    {
        $brand = Brand::with('category')->findOrFail($id);
        return response()->json($brand);
    }

    // ✅ Update brand
    public function updateBrand(Request $request, $id)
    {
        $request->validate([
            'name'        => 'required|string|max:100',
            'image'       => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'status'      => 'required|boolean',
            'category_id' => 'required|integer|exists:categories,id',
        ]);

        $brand = Brand::findOrFail($id);

        $imageUrl = $brand->image; // পুরনো image রাখার জন্য

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('image', 'public');
            $imageUrl = url('storage/' . $imagePath);
        }

        $brand->update([
            'name'        => $request->name,
            'image'       => $imageUrl,
            'status'      => $request->status,
            'category_id' => $request->category_id,
        ]);

        return response()->json([
            'message' => 'Brand updated successfully',
            'data'    => $brand
        ]);
    }

    // ✅ Delete brand
    public function deleteBrand(Request $request)
    {
        $id = $request->id;
        $brand = Brand::findOrFail($id);
        $brand->delete();

        return response()->json([
            'message' => 'Brand deleted successfully'
        ]);
    }

    // ✅ Toggle status
    public function updateBrandStatus(Request $request)
    {
        $request->validate([
            'id'     => 'required|integer|exists:brands,id',
            'status' => 'required|boolean',
        ]);

        $brand = Brand::findOrFail($request->id);
        $brand->status = $request->status;
        $brand->save();

        return response()->json(['message' => 'Status updated successfully']);
    }
}