<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Cart;
use Illuminate\Support\Facades\Log;

class CartController extends Controller
{
    // 🛒 Store Cart Item
    public function storeCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'qty'        => 'required|integer|min:1',
            'price'      => 'required|numeric',
            'size'       => 'nullable|in:XS,S,M,L,XL,XXL', // ✅ validate size
            'color'      => 'nullable|string|max:30',       // ✅ validate color
        ]);

        $client_id = $request->header('id') ?? $request->input('client_id');

        if (!$client_id) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Unauthorized: client_id missing',
            ], 200);
        }

        $exists = Cart::where('client_id', $client_id)
                      ->where('product_id', $request->product_id)
                      ->where('size', $request->size)   // ✅ check variation
                      ->where('color', $request->color) // ✅ check variation
                      ->exists();

        if ($exists) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Product with same variation already in cart',
            ], 200);
        }

        $cart = Cart::create([
            'client_id'  => $client_id,
            'product_id' => $request->product_id,
            'qty'        => $request->qty,
            'price'      => $request->price,
            'size'       => $request->size,
            'color'      => $request->color,
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Cart item created successfully',
            'data'    => $cart,
        ], 201);
    }

    // 🧮 Cart Count by Client_id
    public function cartCount(Request $request)
    {
        $client_id = $request->header('id');

        if (!$client_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client ID missing in header'
            ], 401);
        }

        $count = Cart::where('client_id', $client_id)->count();

        return response()->json([
            'status' => 'success',
            'count' => $count
        ]);
    }

    // 📋 Get Cart Items by Client
    public function getCartByClient(Request $request)
    {
        $client_id = $request->header('id');

        if (!$client_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client ID missing in header'
            ], 401);
        }

        $cartItems = Cart::where('client_id', $client_id)->get();

        return response()->json($cartItems);
    }

    // ❌ Remove Item from Cart
    public function removeFromCart(Request $request)
    {
        try {
            $client_id = $request->header('id');

            if (!$client_id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client ID missing in header'
                ], 401);
            }

            $productId = $request->input('product_id');
            $size      = $request->input('size');   // ✅ optional variation
            $color     = $request->input('color');  // ✅ optional variation

            if (!$productId) {
                return response()->json(['message' => 'Product ID is required'], 400);
            }

            $query = Cart::where('client_id', $client_id)
                         ->where('product_id', $productId);

            if ($size) {
                $query->where('size', $size);
            }
            if ($color) {
                $query->where('color', $color);
            }

            $deleted = $query->delete();

            if ($deleted) {
                return response()->json(['message' => 'Product removed from cart ✅'], 200);
            } else {
                return response()->json(['message' => 'Product not found in cart'], 404);
            }

        } catch (\Exception $e) {
            Log::error('Remove from cart failed: ' . $e->getMessage());
            return response()->json(['message' => 'Server error occurred'], 500);
        }
    }
}