<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use Illuminate\Support\Facades\Log;

class ProductController extends Controller
{
    public function productCreatePage(Request $request)
    {
        return view('admin.product.ProductCreate');
    }

    // 🔍 Search Products
    public function searchProducts(Request $request)
    {
        $query = $request->input('query');

        if (!$query) {
            return response()->json([], 200);
        }

        try {
            $products = Product::where('name', 'LIKE', '%' . $query . '%')
                ->select('id', 'name', 'images') // ✅ use images instead of image
                ->limit(10)
                ->get();

            return response()->json($products, 200);
        } catch (\Exception $e) {
            Log::error('Search failed: ' . $e->getMessage());
            return response()->json(['message' => 'Server error'], 500);
        }
    }

    // 📋 Get all products
    public function productList(Request $request)
    {
        $plans = Product::all();
        return response()->json($plans);
    }

    // 🆕 Store new product
    public function storeProduct(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:100',
            'subtitle' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric',
            'stock' => 'required|integer',
            'discount' => 'nullable|string',
            'tag' => 'nullable|in:Latest,Top Rating,Best Selling,Featured,Special,Gift,Outher',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:brands,id',
            'images.*' => 'required|image|mimes:jpg,jpeg,png|max:2048', // ✅ multiple images
            'status' => 'required|boolean',
        ]);

        $imageUrls = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $imageUrls[] = url('storage/' . $path);
            }
        }

        $plan = Product::create([
            'name' => $request->name,
            'subtitle' => $request->subtitle,
            'description' => $request->description,
            'price' => $request->price,
            'stock' => $request->stock,
            'discount' => $request->discount,
            'tag' => $request->tag,
            'category_id' => $request->category_id,
            'brand_id' => $request->brand_id,
            'images' => json_encode($imageUrls), // ✅ store as JSON
            'status' => $request->status,
        ]);

        return response()->json([
            'message' => 'Product created successfully',
            'data' => $plan
        ], 201);
    }

    // ✏️ Edit single product
    public function editProduct(Request $request, $id)
    {
        $plan = Product::findOrFail($id);
        $plan->images = json_decode($plan->images); // ✅ decode for frontend
        return response()->json($plan);
    }

    // 🔄 Update product
    public function updateProduct(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:100',
            'subtitle' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric',
            'stock' => 'required|integer',
            'discount' => 'nullable|string',
            'tag' => 'nullable|in:Latest,Top Rating,Best Selling,Featured,Special,Gift,Outher',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:brands,id',
            'images.*' => 'image|mimes:jpg,jpeg,png|max:2048', // ✅ multiple images optional
            'status' => 'required|boolean',
        ]);

        $plan = Product::findOrFail($id);

        $imageUrls = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('images', 'public');
                $imageUrls[] = url('storage/' . $path);
            }
        }

        $plan->update([
            'name' => $request->name,
            'subtitle' => $request->subtitle,
            'description' => $request->description,
            'price' => $request->price,
            'stock' => $request->stock,
            'discount' => $request->discount,
            'tag' => $request->tag,
            'images' => json_encode($imageUrls),
            'category_id' => $request->category_id,
            'brand_id' => $request->brand_id,
            'status' => $request->status,
        ]);

        return response()->json([
            'message' => 'Product updated successfully',
            'data' => $plan
        ]);
    }

    // 🗑️ Delete product
    public function deleteProduct(Request $request)
    {
        $id = $request->id;
        $plan = Product::findOrFail($id);
        $plan->delete();

        return response()->json([
            'message' => 'Product deleted successfully'
        ]);
    }

    // 🔀 Toggle status
    public function updateProductStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|integer|exists:products,id',
            'status' => 'required|boolean',
        ]);

        $plan = Product::findOrFail($request->id);
        $plan->status = $request->status;
        $plan->save();

        return response()->json(['message' => 'Status updated successfully']);
    }

    // 🔎 Select product by ID
    public function selectProductById(Request $request, $id)
    {
        $product = Product::findOrFail($id);
        return response()->json($product);
    }

    // 🤝 Related products by brand
    public function relatedByBrand($product_id)
    {
        $product = Product::with('brand')->find($product_id);

        if (!$product || !$product->brand_id) {
            return response()->json(['message' => 'Product or brand not found'], 404);
        }

        $relatedProducts = Product::where('brand_id', $product->brand_id)
            ->where('id', '!=', $product_id)
            ->get();

        return response()->json($relatedProducts);
    }

    // 📦 Get product by ID
    public function getProductById($id)
    {
        $product = Product::find($id);

        if (!$product) {
            return response()->json([
                'message' => 'Product not found.'
            ], 404);
        }

        return response()->json($product);
    }


    public function showProductImage($id)
    {
        $product = Product::findOrFail($id);

        // image_urls ফিল্ডে JSON string আছে
        $imageArray = json_decode($product->images, true); // decode to PHP array

        return response()->json([
            'product_id' => $product->id,
            'images' => $imageArray,
        ]);
    }



}