<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Review;
use App\Models\Product;

class ReviewController extends Controller
{

    // ✅ Get all pricing plans
    public function reviewList(Request $request, $product_id)
    {
           $reviews = Review::with('client') // eager load client info
        ->where('product_id', $product_id)
        ->get()
        ->map(function ($review) {
            return [
                'id' => $review->id,
                'description' => $review->description,
                'rating' => $review->rating,
                'created_at' => $review->Created_at,
                'client_name' => $review->client ? $review->client->first_name . ' ' . $review->client->last_name : 'Anonymous',
            ];
        });

    return response()->json($reviews);

    }

    // getReviewList

    public function getReviewList(Request $request)
    {
        $reviews = Review::all();
        return response()->json($reviews);
    }

    // ✅ Store new pricing plan
    public function storeReview(Request $request)
    {
        // Validation
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'description' => 'required|string',
            'rating' => 'required|integer|min:0|max:5',
        ]);

        // Middleware injected client_id
        $client_id = $request->header('id');

        // Save review
        $review = Review::create([
            'client_id' => $client_id,
            'product_id' => $request->product_id,
            'description' => $request->description,
            'rating' => $request->rating,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Review created successfully',
            'data' => $review
        ], 201);
    }


    // Review Count
    public function reviewCount(Request $request, $product_id)
    {
        $count = Review::where('product_id', $product_id)->count();
        $average = Review::where('product_id', $product_id)->avg('rating');

        return response()->json([
            'count' => $count,
            'averageRating' => $count > 0 ? round($average, 1) : 0
        ]);

    }
    // getReviewByClientID
   public function getReviewByClientID(Request $request)
{
    $client_id = $request->header('id');

    // Get product IDs from reviews
    $productIDs = Review::where('client_id', $client_id)->pluck('product_id');

    // Filter products based on those IDs
    $products = Product::whereIn('id', $productIDs)->get();

    return response()->json($products);
}




    // ✅ Edit single pricing plan
    public function editBrand(Request $request, $id)
    {
        $plan = Review::findOrFail($id);
        $plan->features = json_decode($plan->features); // ✅ decode for frontend
        return response()->json($plan);
    }

    // ✅ Update pricing plan
    public function updateBrand(Request $request, $id)
    {
        $request->validate([
        'name' => 'required|string|max:100',
        'image' => 'required|image|mimes:jpg,jpeg,png|max:2048',
        'status' => 'required|boolean',
    ]);

    if ($request->hasFile('image')) {
        $imagePath = $request->file('image')->store('image', 'public');
        $imageUrl = url('storage/' . $imagePath); // ✅ full URL
    }

    $plan = Review::findOrFail($id);
    $plan->update([
        'name' => $request->name,
        'image' => $imageUrl,
        'status' => $request->status,
    ]);

    return response()->json([
        'message' => 'Pricing plan updated successfully',
        'data' => $plan
    ]);

    }

    // ✅ Delete pricing plan
    public function deleteBrand(Request $request)
    {
        $id = $request->id;
        $plan = Review::findOrFail($id);
        $plan->delete();

        return response()->json([
            'message' => 'Pricing plan deleted successfully'
        ]);
    }

    // ✅ Toggle status
    public function updateBrandStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|integer|exists:brands,id',
            'status' => 'required|boolean',
        ]);

        $plan = Review::findOrFail($request->id);
        $plan->status = $request->status;
        $plan->save();

        return response()->json(['message' => 'Status updated successfully']);
    }
}
