<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Slider;
use App\Models\Category;

class SliderController extends Controller
{
    public function sliderCreatePage(Request $request)
    {
        $categories = Category::where('status', 1)->get(); // ✅ only active categories
        return view('admin.slider.SliderCreate', compact('categories'));

        // return view('admin.slider.SliderCreate');
    }
    

    // ✅ Get all sliders
    public function sliderList(Request $request)
    {
        $sliders = Slider::all();
        return response()->json($sliders);
    }

    // ✅ Store new slider
    public function storeSlider(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:100',
            'subtitle' => 'nullable|string|max:255',
            'image' => 'required|image|mimes:jpg,jpeg,png|max:2048',
            'link' => 'nullable|string|max:255',
            'status' => 'required|boolean',
            'category_id' => 'nullable|integer|exists:categories,id', // ✅ category validation
        ]);

        $imageUrl = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('image', 'public');
            $imageUrl = url('storage/' . $imagePath);
        }

        $slider = Slider::create([
            'title' => $request->input('title'),
            'subtitle' => $request->input('subtitle'),
            'image' => $imageUrl,
            'link' => $request->input('link'),
            'status' => $request->input('status'),
            'category_id' => $request->input('category_id'), // ✅ save category
        ]);

        return response()->json([
            'message' => 'Slider created successfully',
            'data' => $slider
        ], 201);
    }

    // ✅ Edit single slider
    public function editSlider(Request $request, $id)
    {
        $slider = Slider::findOrFail($id);
        return response()->json($slider);
    }

    // ✅ Update slider
    public function updateSlider(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:100',
            'subtitle' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'link' => 'nullable|string|max:255',
            'status' => 'required|boolean',
            'category_id' => 'nullable|integer|exists:categories,id', // ✅ category validation
        ]);

        $slider = Slider::findOrFail($id);

        $imageUrl = $slider->image; // keep old image if not updated
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('image', 'public');
            $imageUrl = url('storage/' . $imagePath);
        }

        $slider->update([
            'title' => $request->title,
            'subtitle' => $request->subtitle,
            'image' => $imageUrl,
            'link' => $request->link,
            'status' => $request->status,
            'category_id' => $request->category_id, // ✅ update category
        ]);

        return response()->json([
            'message' => 'Slider updated successfully',
            'data' => $slider
        ]);
    }

    // ✅ Delete slider
    public function deleteSlider(Request $request)
    {
        $id = $request->id;
        $slider = Slider::findOrFail($id);
        $slider->delete();

        return response()->json([
            'message' => 'Slider deleted successfully'
        ]);
    }

    // ✅ Toggle status
    public function updateSliderStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|integer|exists:sliders,id',
            'status' => 'required|boolean',
        ]);

        $slider = Slider::findOrFail($request->id);
        $slider->status = $request->status;
        $slider->save();

        return response()->json(['message' => 'Status updated successfully']);
    }
}