<?php

namespace Tests\Feature;

use App\Models\Order;
use App\Models\Invoice;
use App\Models\Client;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OrderControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Should return the order with related invoice and client when it exists.
     */
    public function test_show_returns_order_with_relations(): void
    {
        $client = Client::factory()->create();
        $invoice = Invoice::factory()->create(['client_id' => $client->id]);
        $order = Order::factory()->create(['invoice_id' => $invoice->id, 'client_id' => $client->id]);

        $response = $this->get("/orders/{$order->id}");

        $response->assertStatus(200)
            ->assertJsonFragment(['id' => $order->id])
            ->assertJsonStructure([
                'id',
                'invoice' => ['id'],
                'client' => ['id']
            ]);
    }

    /**
     * Should return 404 when the order does not exist.
     */
    public function test_show_returns_404_for_missing_order(): void
    {
        $response = $this->get('/orders/999999');

        $response->assertStatus(404);
    }

    /**
     * Should delete the order and return a success message.
     */
    public function test_destroy_deletes_order_and_returns_message(): void
    {
        $client = Client::factory()->create();
        $invoice = Invoice::factory()->create(['client_id' => $client->id]);
        $order = Order::factory()->create(['invoice_id' => $invoice->id, 'client_id' => $client->id]);

        $response = $this->delete("/orders/{$order->id}");

        $response->assertStatus(200)
            ->assertJsonFragment(['message' => 'Order deleted successfully']);

        $this->assertDatabaseMissing('orders', ['id' => $order->id]);
    }

    /**
     * Should return 404 on delete when the order does not exist.
     */
    public function test_destroy_returns_404_for_missing_order(): void
    {
        $response = $this->delete('/orders/999999');

        $response->assertStatus(404);
    }

    /**
     * Should not include unrelated fields in show response.
     */
    public function test_show_response_has_expected_fields_only(): void
    {
        $client = Client::factory()->create();
        $invoice = Invoice::factory()->create(['client_id' => $client->id]);
        $order = Order::factory()->create(['invoice_id' => $invoice->id, 'client_id' => $client->id]);

        $response = $this->get("/orders/{$order->id}");

        $response->assertStatus(200)
            ->assertJsonPath('id', $order->id)
            ->assertJsonStructure(['id', 'invoice', 'client']);
    }
}
